/* FILE: mreboot.c                              (D. Tottingham  12/26/90)

This is a collection of C functions that provide a reboot facility for xdetect.
All functions have been written and compiled medium model.  The following
functions are included:

r_check_time ()                 compare time with reboot time
r_get_reboot_status ()          get reboot_enabled
r_get_reboot_time ()            get reboot time
r_initialize_reboot ()          initialize reboot structure
r_set_reboot_time ()            set reboot time
r_set_reboot_status ()          set reboot_enabled flag
r_toggle_reboot_status ()       toggle reboot_enabled flag

EXTERNAL FUNCTIONS CALLED:

dm_get_current_time ()          get current time

HISTORY:
   none

*/



/*************************************************************************
                            INCLUDE FILES

*************************************************************************/
#include <stdio.h>

#include "mconst.h"
#include "mdemux.h"
#include "mreboot.h"


/*************************************************************************
                                GLOBALS

*************************************************************************/
PRIVATE R_INFO reboot_info, *r = &reboot_info;
PRIVATE FLAG reboot_enabled;


/*=======================================================================*
 *                            r_check_time                               *
 *=======================================================================*/
/* Compare time with reboot time.  If reboot time has been reached,
   do a warm boot.                                                       */

PUBLIC
FLAG r_check_time ()
{
   static int first_time = TRUE;
   double time;

   if (! reboot_enabled)
      return FALSE;

   if (!first_time)
      r->previous_time = r->current_time;

   /* Convert buffer_start_time into current_time */
   r->current_time = dm_get_current_time();

   if (!first_time) {
      /* Is it time to reset the system?  */
      if (r->previous_time <= r->current_time) {
         if ((r->current_time >= r->reboot_time) &&
             (r->previous_time <= r->reboot_time))
            boot (WARM);

      /* Special case at midnight */
      } else if ((r->current_time >= r->reboot_time) &&
                  (r->previous_time >= r->reboot_time))
            boot (WARM);
   } else first_time = FALSE;
   return ( FALSE);
}

/*=======================================================================*
 *                         r_get_reboot_status                           *
 *=======================================================================*/
/* Get reboot_enabled.                                                   */

PUBLIC
FLAG r_get_reboot_status ()
{
   return reboot_enabled;
}

/*=======================================================================*
 *                          r_get_reboot_time                            *
 *=======================================================================*/
/* Get reboot time.                                                      */

PUBLIC
char * r_get_reboot_time ()
{
   static char out_str[10];

   sprintf (out_str, "%02d:%02d:%02d", r->hour, r->minute, r->second);
   return (out_str);
}

/*=======================================================================*
 *                         r_initialize_reboot                           *
 *=======================================================================*/
/* Initialize reboot structure.                                          */

PUBLIC
void r_initialize_reboot ()
{
   reboot_enabled = REBOOT_ENABLED;
   r->reboot_time = REBOOT_TIME;
}

/*=======================================================================*
 *                          r_set_reboot_time                            *
 *=======================================================================*/
/* Set reboot time.                                                      */

PUBLIC
void r_set_reboot_time (hour, minute, second)
int hour;
int minute;
int second;
{
   r->hour = hour;
   r->minute = minute;
   r->second = second;
   r->reboot_time = ((double) hour) * 3600 + r->minute * 60 + r->second;
}

/*=======================================================================*
 *                        r_set_reboot_status                            *
 *=======================================================================*/
/* Set reboot_enabled flag.                                              */

PUBLIC
void r_set_reboot_status (reboot_status)
FLAG reboot_status;
{
   reboot_enabled = reboot_status;
}

/*=======================================================================*
 *                      r_toggle_reboot_status                           *
 *=======================================================================*/
/* Toggle reboot_enabled flag.                                           */

PUBLIC
void r_toggle_reboot_status ()
{
   reboot_enabled = (reboot_enabled) ? FALSE : TRUE;
}
